// AttentionPay Background Service Worker

const API_URL = 'http://localhost:3000/api';

// State management
let state = {
  isActive: true,
  extensionToken: null,
  prices: {
    recruiters: 10,
    marketers: 5,
    newsletters: 2,
    unknown: 3
  },
  whitelist: [],
  blockedCount: 0
};

// Initialize
chrome.runtime.onInstalled.addListener(() => {
  console.log('AttentionPay extension installed');
  loadState();
  chrome.action.setBadgeBackgroundColor({ color: '#22c55e' });
});

// Load state from storage
async function loadState() {
  return new Promise((resolve) => {
    chrome.storage.local.get(['attentionpay_state'], (result) => {
      if (result.attentionpay_state) {
        state = { ...state, ...result.attentionpay_state };
      }
      resolve();
    });
  });
}

// Handle messages from popup and content scripts
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  handleMessage(message, sender, sendResponse);
  return true; // Keep message channel open for async responses
});

async function handleMessage(message, sender, sendResponse) {
  switch (message.type) {
    case 'CONNECTED':
      state.extensionToken = message.token;
      await syncWithServer();
      break;
      
    case 'DISCONNECTED':
      state.extensionToken = null;
      break;
    
    case 'PRICE_UPDATE':
      state.prices = message.prices;
      saveState();
      broadcastToContentScripts({ type: 'PRICES_CHANGED', prices: state.prices });
      break;
      
    case 'MESSAGE_ACCEPTED':
      handleMessageAccepted(message.message);
      break;
      
    case 'MESSAGE_BLOCKED':
      handleMessageBlocked(message.data);
      break;
      
    case 'GET_STATE':
      sendResponse(state);
      break;
      
    case 'GET_PRICES':
      sendResponse(state.prices);
      break;
      
    case 'CHECK_SENDER':
      const result = await checkSenderWithAPI(message.sender, message.context);
      sendResponse(result);
      break;
  }
}

// Sync with server
async function syncWithServer() {
  if (!state.extensionToken) return;
  
  try {
    const response = await fetch(`${API_URL}/extension/sync`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'X-Extension-Token': state.extensionToken
      }
    });
    
    if (response.ok) {
      const data = await response.json();
      state.prices = data.prices;
      state.whitelist = data.whitelist || [];
      saveState();
    }
  } catch (error) {
    console.error('Sync failed:', error);
  }
}

// Check sender with API
async function checkSenderWithAPI(sender, context) {
  if (!state.extensionToken) {
    // Fallback to local classification
    return checkSenderLocal(sender, context);
  }
  
  try {
    const response = await fetch(`${API_URL}/extension/check`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'X-Extension-Token': state.extensionToken
      },
      body: JSON.stringify({
        senderEmail: sender,
        senderName: sender,
        subject: context,
        platform: 'gmail'
      })
    });
    
    if (response.ok) {
      return await response.json();
    }
  } catch (error) {
    console.error('API check failed:', error);
  }
  
  return checkSenderLocal(sender, context);
}

// Local fallback for sender checking
function checkSenderLocal(sender, context) {
  // Check whitelist first
  if (state.whitelist.some(w => w.email === sender.toLowerCase())) {
    return { action: 'allow', reason: 'whitelisted' };
  }
  
  const classification = classifySender(sender, context);
  const price = state.prices[classification] || state.prices.unknown;
  
  return {
    action: 'block',
    classification: classification,
    price: price,
    reason: 'requires_payment'
  };
}

// Simple sender classification (would use AI in production)
function classifySender(sender, context) {
  const senderLower = sender.toLowerCase();
  const contextLower = (context || '').toLowerCase();
  
  // Recruiter keywords
  const recruiterKeywords = [
    'recruiter', 'recruiting', 'talent', 'hiring', 'opportunity',
    'position', 'role', 'job', 'career', 'headhunter'
  ];
  
  // Marketing keywords
  const marketingKeywords = [
    'marketing', 'promo', 'offer', 'discount', 'sale', 'deal',
    'limited time', 'exclusive', 'special', 'newsletter'
  ];
  
  // Newsletter indicators
  const newsletterKeywords = [
    'newsletter', 'digest', 'weekly', 'daily', 'update',
    'roundup', 'bulletin', 'news', 'subscribe'
  ];
  
  // Check context and sender
  const combined = senderLower + ' ' + contextLower;
  
  if (recruiterKeywords.some(kw => combined.includes(kw))) {
    return 'recruiters';
  }
  
  if (newsletterKeywords.some(kw => combined.includes(kw))) {
    return 'newsletters';
  }
  
  if (marketingKeywords.some(kw => combined.includes(kw))) {
    return 'marketers';
  }
  
  return 'unknown';
}
// Handle accepted message
async function handleMessageAccepted(message) {
  console.log('Message accepted:', message);
  
  // Send to API if connected
  if (state.extensionToken) {
    try {
      await fetch(`${API_URL}/extension/accept`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'X-Extension-Token': state.extensionToken
        },
        body: JSON.stringify({
          senderEmail: message.sender,
          senderName: message.sender,
          category: message.classification || 'unknown',
          platform: message.platform || 'gmail',
          messageSubject: message.preview?.substring(0, 100),
          messagePreview: message.preview,
          price: message.price
        })
      });
    } catch (error) {
      console.error('Failed to record acceptance:', error);
    }
  }
  
  // Show notification
  chrome.notifications.create({
    type: 'basic',
    title: 'Payment Received!',
    message: `You earned $${message.price.toFixed(2)} from ${message.sender}`
  });
}

// Handle blocked message
function handleMessageBlocked(data) {
  state.blockedCount++;
  updateBadge();
  saveState();
  
  console.log('Message blocked:', data);
}

// Update badge with pending count
function updateBadge() {
  chrome.storage.local.get(['attentionpay_state'], (result) => {
    const pendingCount = result.attentionpay_state?.pendingMessages?.length || 0;
    
    if (pendingCount > 0) {
      chrome.action.setBadgeText({ text: pendingCount.toString() });
    } else {
      chrome.action.setBadgeText({ text: '' });
    }
  });
}

// Broadcast message to all content scripts
async function broadcastToContentScripts(message) {
  const tabs = await chrome.tabs.query({});
  
  for (const tab of tabs) {
    try {
      await chrome.tabs.sendMessage(tab.id, message);
    } catch (e) {
      // Tab doesn't have content script, ignore
    }
  }
}

// Save state to storage
function saveState() {
  chrome.storage.local.get(['attentionpay_state'], (result) => {
    const existingState = result.attentionpay_state || {};
    chrome.storage.local.set({
      attentionpay_state: {
        ...existingState,
        prices: state.prices,
        whitelist: state.whitelist,
        blockedCount: state.blockedCount,
        isActive: state.isActive
      }
    });
  });
}

// Periodic check for new messages (would connect to API in production)
chrome.alarms.create('checkMessages', { periodInMinutes: 1 });

chrome.alarms.onAlarm.addListener((alarm) => {
  if (alarm.name === 'checkMessages') {
    updateBadge();
  }
});

console.log('AttentionPay background service worker loaded');
