// AttentionPay Gmail Content Script

(function() {
  'use strict';
  
  console.log('AttentionPay: Gmail content script loaded');
  
  // Configuration
  const CONFIG = {
    checkInterval: 2000,
    selectors: {
      emailRow: 'tr.zA',
      emailSender: '.yW span[email]',
      emailSubject: '.y6 span',
      inboxContainer: '[role="main"]'
    }
  };
  
  let prices = {
    recruiters: 10,
    marketers: 5,
    newsletters: 2,
    unknown: 3
  };
  
  let processedEmails = new Set();
  
  // Get prices from background script
  async function loadPrices() {
    try {
      const response = await chrome.runtime.sendMessage({ type: 'GET_PRICES' });
      if (response) {
        prices = response;
      }
    } catch (e) {
      console.log('AttentionPay: Could not load prices');
    }
  }
  
  // Listen for price updates
  chrome.runtime.onMessage.addListener((message) => {
    if (message.type === 'PRICES_CHANGED') {
      prices = message.prices;
      console.log('AttentionPay: Prices updated', prices);
    }
  });
  
  // Check sender against rules
  async function checkSender(sender, subject) {
    try {
      const response = await chrome.runtime.sendMessage({
        type: 'CHECK_SENDER',
        sender: sender,
        context: subject
      });
      return response;
    } catch (e) {
      return { action: 'allow' };
    }
  }
  
  // Create the attention gate overlay
  function createAttentionGate(emailRow, senderName, classification, price) {
    // Check if already processed
    const emailId = emailRow.getAttribute('data-legacy-thread-id') || 
                    emailRow.querySelector('[data-thread-id]')?.getAttribute('data-thread-id') ||
                    senderName + Date.now();
    
    if (processedEmails.has(emailId)) return;
    processedEmails.add(emailId);
    
    // Create overlay
    const overlay = document.createElement('div');
    overlay.className = 'attentionpay-gate';
    overlay.innerHTML = `
      <div class="attentionpay-gate-content">
        <div class="attentionpay-gate-icon">💰</div>
        <div class="attentionpay-gate-text">
          <strong>${senderName}</strong> wants your attention
        </div>
        <div class="attentionpay-gate-price">
          $${price.toFixed(2)}
        </div>
        <div class="attentionpay-gate-actions">
          <button class="attentionpay-btn attentionpay-btn-accept" data-email-id="${emailId}">
            Accept +$${price.toFixed(2)}
          </button>
          <button class="attentionpay-btn attentionpay-btn-block" data-email-id="${emailId}">
            Block
          </button>
        </div>
      </div>
    `;
    
    // Position relative to email row
    emailRow.style.position = 'relative';
    emailRow.appendChild(overlay);
    
    // Add click handlers
    overlay.querySelector('.attentionpay-btn-accept').addEventListener('click', (e) => {
      e.stopPropagation();
      e.preventDefault();
      handleAccept(emailId, senderName, classification, price, overlay);
    });
    
    overlay.querySelector('.attentionpay-btn-block').addEventListener('click', (e) => {
      e.stopPropagation();
      e.preventDefault();
      handleBlock(emailId, senderName, overlay, emailRow);
    });
  }
  
  function handleAccept(emailId, senderName, classification, price, overlay) {
    // Notify background script
    chrome.runtime.sendMessage({
      type: 'MESSAGE_ACCEPTED',
      message: {
        id: emailId,
        sender: senderName,
        type: classification,
        price: price
      }
    });
    
    // Remove overlay with animation
    overlay.style.opacity = '0';
    setTimeout(() => overlay.remove(), 300);
    
    // Show success indicator
    showToast(`You earned $${price.toFixed(2)}!`, 'success');
  }
  
  function handleBlock(emailId, senderName, overlay, emailRow) {
    // Notify background script
    chrome.runtime.sendMessage({
      type: 'MESSAGE_BLOCKED',
      data: {
        id: emailId,
        sender: senderName
      }
    });
    
    // Hide the email row
    emailRow.style.opacity = '0.3';
    overlay.innerHTML = `
      <div class="attentionpay-gate-content attentionpay-blocked">
        <span>Blocked</span>
      </div>
    `;
  }
  
  function showToast(message, type) {
    const toast = document.createElement('div');
    toast.className = `attentionpay-toast attentionpay-toast-${type}`;
    toast.textContent = message;
    document.body.appendChild(toast);
    
    setTimeout(() => {
      toast.style.opacity = '0';
      setTimeout(() => toast.remove(), 300);
    }, 2000);
  }
  
  // Scan inbox for new emails
  async function scanInbox() {
    const emailRows = document.querySelectorAll(CONFIG.selectors.emailRow);
    
    for (const row of emailRows) {
      // Skip if already has our overlay
      if (row.querySelector('.attentionpay-gate')) continue;
      
      // Get sender info
      const senderEl = row.querySelector(CONFIG.selectors.emailSender);
      const subjectEl = row.querySelector(CONFIG.selectors.emailSubject);
      
      if (!senderEl) continue;
      
      const senderName = senderEl.getAttribute('name') || senderEl.textContent;
      const senderEmail = senderEl.getAttribute('email') || '';
      const subject = subjectEl?.textContent || '';
      
      // Check with background script
      const result = await checkSender(senderEmail || senderName, subject);
      
      if (result.action === 'block') {
        createAttentionGate(row, senderName, result.classification, result.price);
      }
    }
  }
  
  // Initialize
  async function init() {
    await loadPrices();
    
    // Initial scan
    setTimeout(scanInbox, 1000);
    
    // Watch for changes
    const observer = new MutationObserver(() => {
      scanInbox();
    });
    
    // Start observing when inbox loads
    const startObserver = () => {
      const inbox = document.querySelector(CONFIG.selectors.inboxContainer);
      if (inbox) {
        observer.observe(inbox, {
          childList: true,
          subtree: true
        });
        scanInbox();
      } else {
        setTimeout(startObserver, 1000);
      }
    };
    
    startObserver();
    
    // Periodic scan as backup
    setInterval(scanInbox, CONFIG.checkInterval);
  }
  
  // Wait for page to be ready
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }
})();
