// AttentionPay LinkedIn Content Script

(function() {
  'use strict';
  
  console.log('AttentionPay: LinkedIn content script loaded');
  
  // Configuration
  const CONFIG = {
    checkInterval: 2000,
    selectors: {
      messageThread: '.msg-conversation-card',
      messageSender: '.msg-conversation-card__participant-names',
      inMailBadge: '.msg-conversation-card__inbox-badge',
      messageContainer: '.msg-conversations-container__conversations-list'
    }
  };
  
  let prices = {
    recruiters: 10,
    marketers: 5,
    newsletters: 2,
    unknown: 3
  };
  
  let processedMessages = new Set();
  
  // Get prices from background script
  async function loadPrices() {
    try {
      const response = await chrome.runtime.sendMessage({ type: 'GET_PRICES' });
      if (response) {
        prices = response;
      }
    } catch (e) {
      console.log('AttentionPay: Could not load prices');
    }
  }
  
  // Listen for price updates
  chrome.runtime.onMessage.addListener((message) => {
    if (message.type === 'PRICES_CHANGED') {
      prices = message.prices;
    }
  });
  
  // Check if message is from a recruiter
  function isRecruiter(senderName, messagePreview) {
    const recruiterKeywords = [
      'recruiter', 'recruiting', 'talent', 'hiring', 'opportunity',
      'position', 'role', 'career', 'sourcer', 'acquisition'
    ];
    
    const combined = (senderName + ' ' + messagePreview).toLowerCase();
    return recruiterKeywords.some(kw => combined.includes(kw));
  }
  
  // Check if it's an InMail (paid message from LinkedIn)
  function isInMail(messageCard) {
    return messageCard.querySelector(CONFIG.selectors.inMailBadge) !== null;
  }
  
  // Create the attention gate overlay for LinkedIn
  function createAttentionGate(messageCard, senderName, price) {
    const messageId = messageCard.getAttribute('data-conversation-id') || 
                      senderName + Date.now();
    
    if (processedMessages.has(messageId)) return;
    processedMessages.add(messageId);
    
    const overlay = document.createElement('div');
    overlay.className = 'attentionpay-gate attentionpay-gate-linkedin';
    overlay.innerHTML = `
      <div class="attentionpay-gate-content">
        <div class="attentionpay-gate-icon">💼</div>
        <div class="attentionpay-gate-text">
          <strong>${senderName}</strong> wants your attention
        </div>
        <div class="attentionpay-gate-price">$${price.toFixed(2)}</div>
        <div class="attentionpay-gate-actions">
          <button class="attentionpay-btn attentionpay-btn-accept" data-msg-id="${messageId}">
            Accept +$${price.toFixed(2)}
          </button>
          <button class="attentionpay-btn attentionpay-btn-block" data-msg-id="${messageId}">
            Block
          </button>
        </div>
      </div>
    `;
    
    messageCard.style.position = 'relative';
    messageCard.appendChild(overlay);
    
    // Prevent clicking through to message
    overlay.addEventListener('click', (e) => {
      e.stopPropagation();
      e.preventDefault();
    });
    
    overlay.querySelector('.attentionpay-btn-accept').addEventListener('click', (e) => {
      e.stopPropagation();
      e.preventDefault();
      handleAccept(messageId, senderName, price, overlay);
    });
    
    overlay.querySelector('.attentionpay-btn-block').addEventListener('click', (e) => {
      e.stopPropagation();
      e.preventDefault();
      handleBlock(messageId, senderName, overlay, messageCard);
    });
  }
  
  function handleAccept(messageId, senderName, price, overlay) {
    chrome.runtime.sendMessage({
      type: 'MESSAGE_ACCEPTED',
      message: {
        id: messageId,
        sender: senderName,
        type: 'Recruiter • LinkedIn',
        price: price
      }
    });
    
    overlay.style.opacity = '0';
    setTimeout(() => overlay.remove(), 300);
    
    showToast(`You earned $${price.toFixed(2)}!`, 'success');
  }
  
  function handleBlock(messageId, senderName, overlay, messageCard) {
    chrome.runtime.sendMessage({
      type: 'MESSAGE_BLOCKED',
      data: {
        id: messageId,
        sender: senderName,
        platform: 'linkedin'
      }
    });
    
    messageCard.style.opacity = '0.3';
    overlay.innerHTML = `
      <div class="attentionpay-gate-content attentionpay-blocked">
        <span>Blocked</span>
      </div>
    `;
  }
  
  function showToast(message, type) {
    const toast = document.createElement('div');
    toast.className = `attentionpay-toast attentionpay-toast-${type}`;
    toast.textContent = message;
    document.body.appendChild(toast);
    
    setTimeout(() => {
      toast.style.opacity = '0';
      setTimeout(() => toast.remove(), 300);
    }, 2000);
  }
  
  // Scan messages for recruiters
  async function scanMessages() {
    const messageCards = document.querySelectorAll(CONFIG.selectors.messageThread);
    
    for (const card of messageCards) {
      if (card.querySelector('.attentionpay-gate')) continue;
      
      const senderEl = card.querySelector(CONFIG.selectors.messageSender);
      if (!senderEl) continue;
      
      const senderName = senderEl.textContent.trim();
      const isRecruiterMsg = isRecruiter(senderName, card.textContent);
      const isInMailMsg = isInMail(card);
      
      // Gate recruiters and InMails
      if (isRecruiterMsg || isInMailMsg) {
        const price = prices.recruiters;
        createAttentionGate(card, senderName, price);
      }
    }
  }
  
  // Initialize
  async function init() {
    await loadPrices();
    
    setTimeout(scanMessages, 1500);
    
    const observer = new MutationObserver(() => {
      scanMessages();
    });
    
    const startObserver = () => {
      const container = document.querySelector(CONFIG.selectors.messageContainer);
      if (container) {
        observer.observe(container, {
          childList: true,
          subtree: true
        });
        scanMessages();
      } else {
        setTimeout(startObserver, 1000);
      }
    };
    
    startObserver();
    setInterval(scanMessages, CONFIG.checkInterval);
  }
  
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }
})();
