// AttentionPay Popup Script

const API_URL = 'http://localhost:3000/api';

class AttentionPayPopup {
  constructor() {
    this.state = {
      isLoggedIn: false,
      extensionToken: null,
      user: null,
      earnings: {
        today: 0,
        thisMonth: 0,
        total: 0
      },
      prices: {
        recruiters: 10,
        marketers: 5,
        newsletters: 2,
        unknown: 3
      },
      pendingMessages: [],
      isActive: true
    };
    
    this.init();
  }
  
  async init() {
    await this.loadState();
    
    // If logged in, sync with server
    if (this.state.isLoggedIn && this.state.extensionToken) {
      await this.syncWithServer();
    }
    
    this.render();
    this.attachEventListeners();
  }
  
  async loadState() {
    return new Promise((resolve) => {
      chrome.storage.local.get(['attentionpay_state'], (result) => {
        if (result.attentionpay_state) {
          this.state = { ...this.state, ...result.attentionpay_state };
        }
        resolve();
      });
    });
  }
  
  async saveState() {
    return new Promise((resolve) => {
      chrome.storage.local.set({ attentionpay_state: this.state }, resolve);
    });
  }
  
  async syncWithServer() {
    try {
      const response = await fetch(`${API_URL}/extension/sync`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'X-Extension-Token': this.state.extensionToken
        }
      });
      
      if (response.ok) {
        const data = await response.json();
        this.state.user = data.user;
        this.state.prices = data.prices;
        this.state.earnings = data.earnings;
        await this.saveState();
      } else if (response.status === 401) {
        // Token invalid, logout
        this.state.isLoggedIn = false;
        this.state.extensionToken = null;
        await this.saveState();
      }
    } catch (error) {
      console.error('Sync failed:', error);
    }
  }
  
  render() {
    const app = document.getElementById('app');
    
    if (!this.state.isLoggedIn) {
      app.innerHTML = this.renderLoginView();
    } else {
      app.innerHTML = this.renderMainView();
    }
  }
  
  renderLoginView() {
    return `
      <div class="login-container">
        <div class="logo" style="margin-bottom: 24px;">
          <span>attention</span>pay
        </div>
        <h2 class="login-title">Connect your account</h2>
        <p class="login-subtitle">
          1. <a href="http://localhost:3000/login" target="_blank" style="color: #22c55e;">Login to your dashboard</a><br>
          2. Find "Browser Extension" section<br>
          3. Copy your extension token<br>
          4. Paste it below
        </p>
        <input type="text" id="tokenInput" placeholder="Paste your extension token here" 
          style="width: 100%; padding: 12px; border-radius: 8px; border: 1px solid #333; background: #111; color: #fff; margin-bottom: 12px; font-size: 13px;">
        <button class="btn-login" id="connectBtn">
          Connect Account
        </button>
        <p style="font-size: 11px; color: #666; margin-top: 16px;">
          Don't have an account? <a href="http://localhost:3000/register" target="_blank" style="color: #22c55e;">Sign up first</a>
        </p>
      </div>
    `;
  }
  
  renderMainView() {
    return `
      <div class="header">
        <div class="logo"><span>attention</span>pay</div>
        <button class="settings-btn" id="settingsBtn">⚙️</button>
      </div>
      
      <!-- Earnings Card -->
      <div class="earnings-card">
        <div class="earnings-label">This Month</div>
        <div class="earnings-amount">$${this.state.earnings.thisMonth.toFixed(2)}</div>
        <div class="earnings-period">+$${this.state.earnings.today.toFixed(2)} today</div>
        
        <div class="earnings-stats">
          <div class="stat">
            <div class="stat-value">23</div>
            <div class="stat-label">Messages received</div>
          </div>
          <div class="stat">
            <div class="stat-value">89%</div>
            <div class="stat-label">Blocked</div>
          </div>
          <div class="stat">
            <div class="stat-value">$${this.state.earnings.total.toFixed(0)}</div>
            <div class="stat-label">All time</div>
          </div>
        </div>
      </div>
      
      <!-- Price Settings -->
      <div class="section">
        <div class="section-title">Your Attention Prices</div>
        
        <div class="price-item">
          <div class="price-category">
            <div class="price-icon">💼</div>
            <div class="price-name">Recruiters</div>
          </div>
          <div class="price-value">
            <span style="color: #666;">$</span>
            <input type="number" class="price-input" value="${this.state.prices.recruiters}" data-category="recruiters" min="1" max="100">
          </div>
        </div>
        
        <div class="price-item">
          <div class="price-category">
            <div class="price-icon">📢</div>
            <div class="price-name">Marketers</div>
          </div>
          <div class="price-value">
            <span style="color: #666;">$</span>
            <input type="number" class="price-input" value="${this.state.prices.marketers}" data-category="marketers" min="1" max="100">
          </div>
        </div>
        
        <div class="price-item">
          <div class="price-category">
            <div class="price-icon">📧</div>
            <div class="price-name">Newsletters</div>
          </div>
          <div class="price-value">
            <span style="color: #666;">$</span>
            <input type="number" class="price-input" value="${this.state.prices.newsletters}" data-category="newsletters" min="1" max="100">
          </div>
        </div>
        
        <div class="price-item">
          <div class="price-category">
            <div class="price-icon">❓</div>
            <div class="price-name">Unknown</div>
          </div>
          <div class="price-value">
            <span style="color: #666;">$</span>
            <input type="number" class="price-input" value="${this.state.prices.unknown}" data-category="unknown" min="1" max="100">
          </div>
        </div>
      </div>
      
      <!-- Pending Messages -->
      <div class="section">
        <div class="pending-header">
          <div class="section-title" style="margin-bottom: 0;">Pending Messages</div>
          <span class="pending-count">${this.state.pendingMessages.length}</span>
        </div>
        
        ${this.state.pendingMessages.length > 0 
          ? this.state.pendingMessages.map(msg => this.renderMessage(msg)).join('')
          : this.renderEmptyState()
        }
      </div>
      
      <!-- Footer -->
      <div class="footer">
        <div class="status">
          <div class="status-dot ${this.state.isActive ? '' : 'inactive'}"></div>
          <span>${this.state.isActive ? 'Active' : 'Paused'}</span>
        </div>
        <button id="logoutBtn" style="background: none; border: none; color: #666; font-size: 12px; cursor: pointer;">
          Logout
        </button>
        <a href="http://localhost:3000/dashboard" target="_blank" class="dashboard-link">
          Dashboard →
        </a>
      </div>
    `;
  }
  
  renderMessage(msg) {
    return `
      <div class="message-item" data-id="${msg.id}">
        <div class="message-header">
          <div>
            <div class="message-sender">${msg.sender}</div>
            <div class="message-type">${msg.type}</div>
          </div>
          <div class="message-price">$${msg.price.toFixed(2)}</div>
        </div>
        <div class="message-preview">${msg.preview}</div>
        <div class="message-actions">
          <button class="btn btn-accept" data-action="accept" data-id="${msg.id}">
            Accept +$${msg.price.toFixed(2)}
          </button>
          <button class="btn btn-reject" data-action="reject" data-id="${msg.id}">
            Reject
          </button>
        </div>
      </div>
    `;
  }
  
  renderEmptyState() {
    return `
      <div class="empty-state">
        <div class="empty-icon">✨</div>
        <div>No pending messages</div>
        <div style="font-size: 11px; margin-top: 4px;">Your inbox is protected</div>
      </div>
    `;
  }
  
  attachEventListeners() {
    // Connect button
    const connectBtn = document.getElementById('connectBtn');
    if (connectBtn) {
      connectBtn.addEventListener('click', () => this.handleConnect());
    }
    
    // Settings button
    const settingsBtn = document.getElementById('settingsBtn');
    if (settingsBtn) {
      settingsBtn.addEventListener('click', () => this.openSettings());
    }
    
    // Logout button
    const logoutBtn = document.getElementById('logoutBtn');
    if (logoutBtn) {
      logoutBtn.addEventListener('click', () => this.handleLogout());
    }
    
    // Price inputs
    document.querySelectorAll('.price-input').forEach(input => {
      input.addEventListener('change', (e) => this.handlePriceChange(e));
    });
    
    // Message actions
    document.querySelectorAll('[data-action]').forEach(btn => {
      btn.addEventListener('click', (e) => this.handleMessageAction(e));
    });
  }
  
  async handleConnect() {
    const tokenInput = document.getElementById('tokenInput');
    const token = tokenInput?.value?.trim();
    
    if (!token) {
      alert('Please enter your extension token');
      return;
    }
    
    try {
      const response = await fetch(`${API_URL}/extension/sync`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'X-Extension-Token': token
        }
      });
      
      if (response.ok) {
        const data = await response.json();
        this.state.isLoggedIn = true;
        this.state.extensionToken = token;
        this.state.user = data.user;
        this.state.prices = data.prices;
        this.state.earnings = data.earnings;
        
        await this.saveState();
        
        // Notify background script
        chrome.runtime.sendMessage({
          type: 'CONNECTED',
          token: token
        });
        
        this.render();
        this.attachEventListeners();
      } else {
        alert('Invalid token. Please check your token and try again.');
      }
    } catch (error) {
      console.error('Connection failed:', error);
      alert('Connection failed. Please try again.');
    }
  }
  
  async handleLogout() {
    this.state.isLoggedIn = false;
    this.state.extensionToken = null;
    this.state.user = null;
    this.state.earnings = { today: 0, thisMonth: 0, total: 0 };
    this.state.pendingMessages = [];
    
    await this.saveState();
    
    chrome.runtime.sendMessage({ type: 'DISCONNECTED' });
    
    this.render();
    this.attachEventListeners();
  }
  
  async handleLogin() {
    // Legacy - not used anymore
  }
  
  async legacyLogin() {
    // For demo, just set logged in
    // In production, this would trigger OAuth flow
    this.state.isLoggedIn = true;
    this.state.user = {
      name: 'Demo User',
      email: 'demo@example.com'
    };
    
    // Add some demo pending messages
    this.state.pendingMessages = [
      {
        id: '1',
        sender: 'TechCorp Recruiting',
        type: 'Recruiter • LinkedIn',
        preview: 'Hi! I came across your profile and think you would be a great fit for our Senior Engineer role...',
        price: 10.00
      },
      {
        id: '2', 
        sender: 'SaaS Weekly',
        type: 'Newsletter • Email',
        preview: 'This week in SaaS: Top funding rounds, product launches, and industry insights...',
        price: 2.00
      }
    ];
    
    await this.saveState();
    this.render();
    this.attachEventListeners();
  }
  
  async handlePriceChange(e) {
    const category = e.target.dataset.category;
    const value = parseFloat(e.target.value) || 1;
    
    this.state.prices[category] = Math.max(1, Math.min(100, value));
    e.target.value = this.state.prices[category];
    
    await this.saveState();
    
    // Notify background script of price change
    chrome.runtime.sendMessage({
      type: 'PRICE_UPDATE',
      prices: this.state.prices
    });
  }
  
  async handleMessageAction(e) {
    const action = e.target.dataset.action;
    const messageId = e.target.dataset.id;
    const message = this.state.pendingMessages.find(m => m.id === messageId);
    
    if (!message) return;
    
    if (action === 'accept') {
      // Add to earnings
      this.state.earnings.today += message.price;
      this.state.earnings.thisMonth += message.price;
      this.state.earnings.total += message.price;
      
      // Show notification
      chrome.runtime.sendMessage({
        type: 'MESSAGE_ACCEPTED',
        message: message
      });
    }
    
    // Remove from pending
    this.state.pendingMessages = this.state.pendingMessages.filter(m => m.id !== messageId);
    
    await this.saveState();
    this.render();
    this.attachEventListeners();
  }
  
  openSettings() {
    chrome.tabs.create({ url: 'https://attentionpay.io/settings' });
  }
}

// Initialize
document.addEventListener('DOMContentLoaded', () => {
  new AttentionPayPopup();
});
